<?php
/**
 * Plugin Name: Product Shortfacts for WooCommerce
 * Description: Add up to 10 key product shortfacts and display them as bullet points or checkmarks on product pages with configurable styles.
 * Version: 2.0.2
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * Author: wefathemes
 * Author URI: https://wefa.digital
 * Text Domain: product-shortfacts-for-woocommerce
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define constants (use PSF to avoid collisions with other plugins)
define('PSF_PLUGIN_VERSION', '2.0.2');
define('PSF_PLUGIN_FILE', __FILE__);
define('PSF_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PSF_PLUGIN_URL', plugin_dir_url(__FILE__));

// Ensure WooCommerce is active
function psf_is_woocommerce_active(): bool {
    return in_array('woocommerce/woocommerce.php', (array) get_option('active_plugins', []), true)
        || (function_exists('is_plugin_active') && is_plugin_active('woocommerce/woocommerce.php'));
}

// Activation: basic checks
function psf_activate() {
    if (!psf_is_woocommerce_active()) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(esc_html__('WooCommerce must be active to use Product Shortfacts.', 'product-shortfacts-for-woocommerce'));
    }
    // Initialize registration date if not set
    if (!get_option('psf_registration_date')) {
        update_option('psf_registration_date', current_time('timestamp'));
    }
}
register_activation_hook(__FILE__, 'psf_activate');

// Deactivate after 30-day trial ends
function psf_maybe_deactivate_if_expired() {
    $reg = (int) get_option('psf_registration_date');
    $now = current_time('timestamp');
    $days = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0; // Missing date treated as not expired
    if ($days >= 30) {
        // If license already valid, nothing to do
        $licensed = get_option('psf_license_valid') === 'yes';
        if ($licensed) {
            return;
        }
        // Show a non-blocking admin notice encouraging license entry (modal will auto-open)
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>' . esc_html__('Product Shortfacts trial has expired. Please enter your license key to continue using the plugin.', 'product-shortfacts-for-woocommerce') . '</p></div>';
        });
    }
}
add_action('admin_init', 'psf_maybe_deactivate_if_expired');

// Autoload simple includes
require_once PSF_PLUGIN_DIR . 'includes/helpers.php';
require_once PSF_PLUGIN_DIR . 'includes/admin.php';
require_once PSF_PLUGIN_DIR . 'includes/frontend.php';

// i18n: WordPress loads translations automatically from language packs.
// No manual textdomain loading is needed or recommended.

// Admin init
add_action('plugins_loaded', function () {
    if (!psf_is_woocommerce_active()) {
        return;
    }
    // Migrate legacy heading text from older plugin versions
    $heading = get_option('psf_heading_text', '');
    if ($heading === 'Buy on Marketplaces') {
        update_option('psf_heading_text', 'Key Features');
    }
    \PSF\Admin::init();
    // Initialize frontend when licensed OR within trial period
    $licensed = get_option('psf_license_valid') === 'yes';
    $reg = (int) get_option('psf_registration_date');
    $now = current_time('timestamp');
    $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
    $expired = ($days_used >= 30);
    if ($licensed || !$expired) {
        \PSF\Frontend::init();
    }
});

// Ensure defaults if missing (handles cases where activation hook did not set options)
function psf_ensure_defaults() {
    // Create license key option if missing
    $license_opt = get_option('psf_license_key', null);
    if ($license_opt === null) {
        add_option('psf_license_key', '');
    }
    // Ensure registration date if missing (in case activation was interrupted)
    if (!get_option('psf_registration_date')) {
        update_option('psf_registration_date', current_time('timestamp'));
    }
}
add_action('admin_init', 'psf_ensure_defaults');

// Migrate legacy EML option keys to PSF-prefixed keys
function psf_migrate_settings_options() {
    $map = [
        'eml_license_key' => 'psf_license_key',
        'eml_license_valid' => 'psf_license_valid',
        'eml_license_last_status' => 'psf_license_last_status',
        'eml_license_last_checked' => 'psf_license_last_checked',
        'eml_registration_date' => 'psf_registration_date',
        'eml_registration_date_display' => 'psf_registration_date_display',
        'eml_trial_days_left' => 'psf_trial_days_left',
        'eml_support_email_display' => 'psf_support_email_display',
        'eml_text_size' => 'psf_text_size',
        'eml_bullet_style' => 'psf_bullet_style',
        'eml_shortfacts_limit' => 'psf_shortfacts_limit',
        'eml_heading_text' => 'psf_heading_text',
        'eml_show_heading' => 'psf_show_heading',
    ];
    foreach ($map as $old => $new) {
        $new_val = get_option($new, null);
        if ($new_val === null) {
            $old_val = get_option($old, null);
            if ($old_val !== null) {
                add_option($new, $old_val);
            }
        }
    }
}
add_action('admin_init', 'psf_migrate_settings_options');

// Helper: is plugin licensed
function psf_is_licensed(): bool {
    $key = trim((string) get_option('psf_license_key'));
    if ($key === '') {
        return false;
    }
    return get_option('psf_license_valid') === 'yes';
}

// Sync: if license key becomes empty, ensure valid flag is reset
function psf_sync_license_valid_flag() {
    $key = trim((string) get_option('psf_license_key'));
    if ($key === '' && get_option('psf_license_valid') === 'yes') {
        update_option('psf_license_valid', 'no');
    }
}
add_action('init', 'psf_sync_license_valid_flag');
add_action('admin_init', 'psf_sync_license_valid_flag');

// License validation
function psf_check_license_valid_now(): ?bool {
    $license_raw = trim((string) get_option('psf_license_key'));
    // Normalize license to uppercase to match expected format on server
    $license = strtoupper($license_raw);
    // Basic format validation before remote call
    if ($license === '' || !preg_match('/^WEFAEML-[A-Z0-9]{8}-[A-Z0-9]{8}-[A-Z0-9]{8}-[A-Z0-9]{8}$/', $license)) {
        update_option('psf_license_last_status', 'invalid');
        update_option('psf_license_last_checked', current_time('timestamp'));
        update_option('psf_license_valid', 'no');
        return false;
    }
    $url = add_query_arg(['license' => rawurlencode($license)], 'https://wefa.digital/wp_psf_licence.check.php');
    $resp = wp_remote_get($url, [
        'timeout' => 5,
        'redirection' => 3,
        'headers' => [ 'Accept' => 'text/plain' ],
    ]);
    if (is_wp_error($resp)) {
        update_option('psf_license_last_status', 'skipped');
        update_option('psf_license_last_checked', current_time('timestamp'));
        return null;
    }
    $code = wp_remote_retrieve_response_code($resp);
    $body = trim((string) wp_remote_retrieve_body($resp));
    update_option('psf_license_last_checked', current_time('timestamp'));
    if ((int) $code === 200 && $body === 'valid licence !') {
        update_option('psf_license_last_status', 'valid');
        update_option('psf_license_valid', 'yes');
        return true;
    } elseif ((int) $code === 403 || $body === 'invalid licence') {
        update_option('psf_license_last_status', 'invalid');
        update_option('psf_license_valid', 'no');
        return false;
    }
    update_option('psf_license_last_status', 'skipped');
    return null;
}

function psf_maybe_check_license_daily() {
    $license = trim((string) get_option('psf_license_key'));
    if (empty($license)) {
        return;
    }
    $last = (int) get_option('psf_license_last_checked');
    $now = current_time('timestamp');
    if (!$last || ($now - $last) >= DAY_IN_SECONDS) {
        psf_check_license_valid_now();
    }
}
add_action('admin_init', 'psf_maybe_check_license_daily');

