<?php
namespace PSF;

if (!defined('ABSPATH')) {
    exit;
}

class Admin
{
    public static function init(): void
    {
        \add_action('add_meta_boxes', [self::class, 'add_product_metabox']);
        \add_action('save_post_product', [self::class, 'save_product_shortfacts'], 10, 2);
        \add_action('admin_enqueue_scripts', [self::class, 'enqueue_admin_assets']);

        // License prompt and handler
        \add_action('admin_notices', [self::class, 'render_license_prompt']);
        // Show post-submit license status notice regardless of license state
        \add_action('admin_notices', [self::class, 'show_license_status_notice']);
        \add_action('admin_post_psf_save_license', [self::class, 'handle_license_submit']);

        // Modal overlay printed in admin footer when unlicensed
        \add_action('admin_footer', [self::class, 'render_license_modal']);

        \add_filter('woocommerce_settings_tabs_array', [self::class, 'register_settings_tab'], 50);
        \add_action('woocommerce_settings_tabs_psf_settings', [self::class, 'render_settings']);
        \add_action('woocommerce_update_options_psf_settings', [self::class, 'save_settings']);
    }

    public static function render_license_prompt(): void
    {
        // Show prompt to admins until licensed
        if (\function_exists('psf_is_licensed') && \psf_is_licensed()) {
            return;
        }
        // Honor free trial: only prompt when expired
        $reg = (int) \get_option('psf_registration_date');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $expired = ($days_used >= 30);
        if (!$expired) {
            return;
        }
        if (!\current_user_can('manage_options')) {
            return;
        }
        $license_val = (string) \get_option('psf_license_key', '');
        $action_url = \admin_url('admin-post.php');
        echo '<div class="notice notice-error"><form method="post" action="' . \esc_url($action_url) . '">';
        echo '<input type="hidden" name="action" value="psf_save_license" />';
        \wp_nonce_field('psf_save_license', 'psf_save_license_nonce');
        echo '<p><strong>' . \esc_html__('Product Shortfacts requires a license key', 'product-shortfacts-for-woocommerce') . '</strong></p>';
        echo '<p><label for="psf_license_key_inline">' . \esc_html__('License Key', 'product-shortfacts-for-woocommerce') . '</label> ';
        echo '<input id="psf_license_key_inline" type="text" name="psf_license_key" class="regular-text" value="' . \esc_attr($license_val) . '" /> ';
        echo '<button type="submit" class="button button-primary">' . \esc_html__('Save and Validate', 'product-shortfacts-for-woocommerce') . '</button></p>';
        echo '<p>' . \esc_html__('You can buy a license at wefa.digital. This prompt will continue to appear until a valid license is saved.', 'product-shortfacts-for-woocommerce') . '</p>';
        echo '</form></div>';
    }

    public static function handle_license_submit(): void
    {
        if (!\current_user_can('manage_options')) {
            \wp_die('Unauthorized');
        }
        $nonce_val = \filter_input(INPUT_POST, 'psf_save_license_nonce');
        $nonce_val = \is_string($nonce_val) ? \sanitize_text_field(\wp_unslash($nonce_val)) : '';
        if (empty($nonce_val) || !\wp_verify_nonce($nonce_val, 'psf_save_license')) {
            \wp_die('Invalid nonce');
        }
        $key_raw = \filter_input(INPUT_POST, 'psf_license_key');
        $key = \sanitize_text_field(\is_string($key_raw) ? \wp_unslash($key_raw) : '');
        $key = strtoupper(trim($key));
        \update_option('psf_license_key', $key);
        $result = \psf_check_license_valid_now();
        $status = ($result === true) ? 'valid' : (($result === false) ? 'invalid' : 'skipped');
        if ($status === 'valid') {
            \update_option('psf_license_valid', 'yes');
        } elseif ($status === 'invalid') {
            \update_option('psf_license_valid', 'no');
        }
        // Persist status temporarily to ensure a notice even if query args are stripped
        \set_transient('psf_license_last_notice', $status, 300);
        $redirect = \wp_get_referer();
        if (!$redirect) {
            $redirect = \admin_url();
        }
        // Surface a clear admin notice on the next page
        \wp_safe_redirect(\add_query_arg(['psf_license_status' => $status], $redirect));
        exit;
    }

    public static function enqueue_admin_assets(): void
    {
        \wp_enqueue_style('psf-admin', PSF_PLUGIN_URL . 'assets/admin.css', [], PSF_PLUGIN_VERSION);
        \wp_enqueue_script('psf-admin', PSF_PLUGIN_URL . 'assets/admin.js', ['jquery'], PSF_PLUGIN_VERSION, true);
        $licensed = \function_exists('psf_is_licensed') ? \psf_is_licensed() : false;
        $reg = (int) \get_option('psf_registration_date');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $expired = ($days_used >= 30);
        $status_raw = \filter_input(INPUT_GET, 'psf_license_status');
        $status = \sanitize_text_field(\is_string($status_raw) ? \wp_unslash($status_raw) : '');
        \wp_localize_script('psf-admin', 'PSFAdmin', [
            'licensed' => $licensed,
            'expired' => $expired,
            'status' => $status,
        ]);
    }

    public static function add_product_metabox(): void
    {
        // Show metabox when licensed OR within trial period
        $reg = (int) \get_option('psf_registration_date');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $expired = ($days_used >= 30);
        if (\function_exists('psf_is_licensed') && !\psf_is_licensed() && $expired) {
            return;
        }
        \add_meta_box(
            'psf_product_shortfacts',
            \__('Product Shortfacts', 'product-shortfacts-for-woocommerce'),
            [self::class, 'render_product_metabox'],
            'product',
            'normal',
            'default'
        );
    }

    public static function render_product_metabox($post): void
    {
        \wp_nonce_field('psf_save_shortfacts', 'psf_nonce');
        $facts = \get_post_meta($post->ID, '_psf_shortfacts', true);
        if (!is_array($facts) || empty($facts)) {
            // Fallback to legacy key
            $facts = \get_post_meta($post->ID, '_eml_shortfacts', true);
        }
        if (!is_array($facts)) {
            $facts = [];
        }
        $limit = (int) \get_option('psf_shortfacts_limit', 10);
        if ($limit < 1) { $limit = 10; }
        echo '<div class="eml-shortfacts-container">';
        echo '<p class="description">' . \esc_html__('Add up to 10 key product shortfacts (one per line).', 'product-shortfacts-for-woocommerce') . '</p>';
        echo '<div class="eml-shortfacts-grid">';
        for ($i = 0; $i < $limit; $i++) {
            $value = isset($facts[$i]) ? $facts[$i] : '';
            echo '<div class="eml-item">';
            /* translators: %d: The shortfact number */
            echo '<label for="psf_fact_' . esc_attr($i) . '"><strong>' . \esc_html(sprintf(__('Shortfact #%d', 'product-shortfacts-for-woocommerce'), $i + 1)) . '</strong></label>';
            echo '<input type="text" class="regular-text" id="psf_fact_' . esc_attr($i) . '" name="psf_shortfacts[' . esc_attr($i) . ']" value="' . esc_attr($value) . '" placeholder="' . \esc_attr__('e.g., Battery lasts up to 12 hours', 'product-shortfacts-for-woocommerce') . '" maxlength="200" />';
            echo '</div>';
        }
        echo '</div>';
        echo '</div>';
    }

    public static function render_license_modal(): void
    {
        if (\function_exists('psf_is_licensed') && \psf_is_licensed()) {
            return;
        }
        // Honor free trial: only render modal when expired
        $reg = (int) \get_option('psf_registration_date');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $expired = ($days_used >= 30);
        if (!$expired) {
            return;
        }
        if (!\current_user_can('manage_options')) {
            return;
        }
        $action_url = \admin_url('admin-post.php');
        $license_val = (string) \get_option('psf_license_key', '');
        echo '<div id="eml-modal-overlay" class="eml-modal-overlay" aria-hidden="true">';
        echo '<div class="eml-modal" role="dialog" aria-labelledby="eml-modal-title" aria-modal="true">';
        echo '<div class="eml-modal-header">';
        // Add product heading above the license entry title
        echo '<div class="eml-title-group">';
        echo '<h3 class="eml-modal-subtitle">' . \esc_html__('Product Shortfacts Pro', 'product-shortfacts-for-woocommerce') . '</h3>';
        echo '<h2 id="eml-modal-title">' . \esc_html__('Enter License Key', 'product-shortfacts-for-woocommerce') . '</h2>';
        echo '</div>';
        echo '<button type="button" id="eml-modal-close" class="button">' . \esc_html__('Close', 'product-shortfacts-for-woocommerce') . '</button>';
        echo '</div>';
        echo '<div class="eml-modal-body">';
        // Inline feedback inside modal based on last license attempt
        $st_raw = \filter_input(INPUT_GET, 'psf_license_status');
        if (\is_string($st_raw)) {
            $st = \sanitize_text_field(\wp_unslash($st_raw));
            if ($st === 'valid') {
                echo '<div class="notice notice-success"><p>' . \esc_html__('License validated successfully.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            } elseif ($st === 'invalid') {
                echo '<div class="notice notice-error"><p>' . \esc_html__('Invalid license key. Please check and try again.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            } elseif ($st === 'skipped') {
                echo '<div class="notice notice-warning"><p>' . \esc_html__('License check skipped (network issue). Will retry later.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            }
        }
        echo '<p>' . \esc_html__('Your trial has expired. Enter your license key to continue using Product Shortfacts.', 'product-shortfacts-for-woocommerce') . '</p>';
        echo '<form method="post" action="' . \esc_url($action_url) . '" class="eml-license-form">';
        echo '<input type="hidden" name="action" value="psf_save_license" />';
        \wp_nonce_field('psf_save_license', 'psf_save_license_nonce');
        echo '<label for="psf_license_key_modal"><strong>' . \esc_html__('License Key', 'product-shortfacts-for-woocommerce') . '</strong></label>';
        echo '<input id="psf_license_key_modal" type="text" name="psf_license_key" class="regular-text" value="' . \esc_attr($license_val) . '" placeholder="WEFAEML-XXXX-XXXX-XXXX-XXXX" />';
        echo '<div class="eml-modal-actions">';
        echo '<button type="submit" class="button button-primary">' . \esc_html__('Save and Validate', 'product-shortfacts-for-woocommerce') . '</button>';
        echo '<a href="https://wefa.digital/wp-ext-marketplaces/" target="_blank" rel="noopener" class="button btn-danger">' . \esc_html__('Buy License', 'product-shortfacts-for-woocommerce') . '</a>';
        echo '</div>';
        echo '</form>';
        echo '</div>';
        echo '</div>';
        echo '</div>';
    }

    public static function save_product_shortfacts($post_id, $post): void
    {
        // Allow saving during trial; block only if expired and unlicensed
        $reg = (int) \get_option('psf_registration_date');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $expired = ($days_used >= 30);
        if (\function_exists('psf_is_licensed') && !\psf_is_licensed() && $expired) {
            return;
        }
        $nonce_links = \filter_input(INPUT_POST, 'psf_nonce');
        $nonce_links = \is_string($nonce_links) ? \sanitize_text_field(\wp_unslash($nonce_links)) : '';
        if (empty($nonce_links) || !\wp_verify_nonce($nonce_links, 'psf_save_shortfacts')) {
            return;
        }
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if ($post->post_type !== 'product') {
            return;
        }
        if (!\current_user_can('edit_post', $post_id)) {
            return;
        }
        $raw = \filter_input(INPUT_POST, 'psf_shortfacts', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY);
        $raw = \is_array($raw) ? \wp_unslash($raw) : [];
        $limit = (int) \get_option('psf_shortfacts_limit', 10);
        if ($limit < 1) { $limit = 10; }
        $sanitized = [];
        for ($i = 0; $i < $limit; $i++) {
            $value = isset($raw[$i]) ? \sanitize_text_field($raw[$i]) : '';
            $value = trim($value);
            if ($value !== '') {
                // Ensure max length
                $value = mb_substr($value, 0, 200);
                $sanitized[$i] = $value;
            }
        }
        \update_post_meta($post_id, '_psf_shortfacts', $sanitized);
    }

    public static function register_settings_tab($tabs)
    {
        $tabs['psf_settings'] = \__('Product Shortfacts', 'product-shortfacts-for-woocommerce');
        return $tabs;
    }

    public static function render_settings(): void
    {
        echo '<div class="eml-settings eml-card">';
        echo '<h2>' . \esc_html__('Product Shortfacts Settings', 'product-shortfacts-for-woocommerce') . '</h2>';

        // Status after inline save
        $st_raw_settings = \filter_input(INPUT_GET, 'psf_license_status');
        if (\is_string($st_raw_settings)) {
            $st = \sanitize_text_field(\wp_unslash($st_raw_settings));
            if ($st === 'valid') {
                echo '<div class="notice notice-success"><p>' . \esc_html__('License validated successfully.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            } elseif ($st === 'invalid') {
                echo '<div class="notice notice-error"><p>' . \esc_html__('Invalid license key. Please check and try again.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            } else {
                echo '<div class="notice notice-warning"><p>' . \esc_html__('License check skipped (network issue). Will retry later.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            }
        }

        // Trial info and license
        $reg = (int) \get_option('psf_registration_date');
        $license = (string) \get_option('psf_license_key');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $days_left = max(0, 30 - $days_used);
        $license_status = \get_option('psf_license_last_status');
        $license_valid = \get_option('psf_license_valid') === 'yes';
        $expired = ($days_used >= 30);
        $is_licensed = (\function_exists('psf_is_licensed')) ? \psf_is_licensed() : $license_valid;

        if (! $is_licensed && $expired) {
            if ($expired) {
                echo '<div class="notice notice-error"><p>' . \esc_html__('Your trial has expired. Enter your license key to unlock Product Shortfacts.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            } else {
                echo '<div class="notice notice-warning"><p>' . \esc_html__('Please enter a valid license key to unlock Product Shortfacts.', 'product-shortfacts-for-woocommerce') . '</p></div>';
            }
        } else {
            echo '<div class="eml-settings-row">';
            echo '<div class="eml-col">';
            echo '<p><strong>' . \esc_html__('Registration Date:', 'product-shortfacts-for-woocommerce') . '</strong> ' . ($reg ? \esc_html(\date_i18n(\get_option('date_format'), $reg)) : \esc_html__('Not set', 'product-shortfacts-for-woocommerce')) . '</p>';
            // Always show trial status when unlicensed
            if (! $is_licensed) {
                echo '<p><strong>' . \esc_html__('Trial Status:', 'product-shortfacts-for-woocommerce') . '</strong> ' . (
                    $expired
                        ? \esc_html__('Expired', 'product-shortfacts-for-woocommerce')
                        /* translators: %d: Number of days left in trial */
                        : \esc_html(\sprintf(\__('Active (%d days left)', 'product-shortfacts-for-woocommerce'), $days_left))
                ) . '</p>';
            }
            // Hide Trial Days Left once licensed
            if (!empty($license)) {
                $status_label = $license_valid ? \__('Valid', 'product-shortfacts-for-woocommerce')
                    : ($license_status === 'invalid' ? \__('Invalid', 'product-shortfacts-for-woocommerce')
                        : ($license_status === 'skipped' ? \__('Skipped today', 'product-shortfacts-for-woocommerce')
                            : \__('Unknown', 'product-shortfacts-for-woocommerce')));
                echo '<p><strong>' . \esc_html__('License Status:', 'product-shortfacts-for-woocommerce') . '</strong> ' . \esc_html($status_label) . '</p>';
            }
            echo '<p><strong>' . \esc_html__('Support:', 'product-shortfacts-for-woocommerce') . '</strong> <a href="mailto:shop@wefa.digital">shop@wefa.digital</a></p>';
            echo '</div>';
            echo '<div class="eml-col">';
            echo '<a class="button button-primary" target="_blank" rel="noopener" href="https://wefa.digital">' . \esc_html__('Buy here on wefa.digital', 'product-shortfacts-for-woocommerce') . '</a>';
            echo '</div>';
            echo '</div>';
        }

        // Keep display fields synced
        $formatted_date = $reg ? \date_i18n(\get_option('date_format'), $reg) : \__('Not set', 'product-shortfacts-for-woocommerce');
        \update_option('psf_registration_date_display', $formatted_date);
        \update_option('psf_trial_days_left', $license_valid ? '' : (string) $days_left);
        \update_option('psf_support_email_display', 'shop@wefa.digital');

        // Render settings fields
        $fields = self::get_settings_fields();
        // Hide Trial Days Left field entirely once licensed
        if ($is_licensed && isset($fields['psf_trial_days_left'])) {
            unset($fields['psf_trial_days_left']);
        }
        if (! $is_licensed && $expired) {
            // Hide all configuration; show only license field when unlicensed
            $fields = [
                'section_title' => [
                    'name' => \__('Product Shortfacts', 'product-shortfacts-for-woocommerce'),
                    'type' => 'title',
                    'desc' => \__('Enter your license key to unlock the plugin.', 'product-shortfacts-for-woocommerce'),
                    'id' => 'psf_section_title',
                ],
                'psf_license_key' => $fields['psf_license_key'],
            ];
        }
        \woocommerce_admin_fields($fields);
        echo '</div>';
    }

    public static function save_settings(): void
    {
        // Persist WooCommerce settings
        \woocommerce_update_options(self::get_settings_fields());
        // Immediately validate license if a key is present after save
        $key = (string) \get_option('psf_license_key');
        $key = strtoupper(trim($key));
        if ($key !== '') {
            $result = \psf_check_license_valid_now();
            $status = ($result === true) ? 'valid' : (($result === false) ? 'invalid' : 'skipped');
            // Ensure flag coherence for UI
            if ($status === 'valid') {
                \update_option('psf_license_valid', 'yes');
            } elseif ($status === 'invalid') {
                \update_option('psf_license_valid', 'no');
            }
            // Surface an admin notice on next page load
            \set_transient('psf_license_last_notice', $status, 300);
        } else {
            // No key: reflect as unknown in UI and reset flag
            \update_option('psf_license_valid', 'no');
            \update_option('psf_license_last_status', '');
        }
    }

    private static function get_settings_fields(): array
    {
        return [
            'section_title' => [
                'name' => \__('Product Shortfacts', 'product-shortfacts-for-woocommerce'),
                'type' => 'title',
                'desc' => \__('Configure how shortfacts are displayed on product pages.', 'product-shortfacts-for-woocommerce'),
                'id' => 'psf_section_title',
            ],
            'psf_registration_date_display' => [
                'name' => \__('Registration Date', 'product-shortfacts-for-woocommerce'),
                'type' => 'text',
                'id' => 'psf_registration_date_display',
                'desc_tip' => true,
                'desc' => \__('Automatically set on activation.', 'product-shortfacts-for-woocommerce'),
                'custom_attributes' => [ 'disabled' => 'disabled' ],
            ],
            'psf_trial_days_left' => [
                'name' => \__('Trial Days Left', 'product-shortfacts-for-woocommerce'),
                'type' => 'text',
                'id' => 'psf_trial_days_left',
                'custom_attributes' => [ 'disabled' => 'disabled' ],
            ],
            'psf_support_email_display' => [
                'name' => \__('Support Email', 'product-shortfacts-for-woocommerce'),
                'type' => 'text',
                'id' => 'psf_support_email_display',
                'custom_attributes' => [ 'disabled' => 'disabled' ],
            ],
            'psf_license_key' => [
                'name' => \__('License Key', 'product-shortfacts-for-woocommerce'),
                'type' => 'text',
                'desc' => \__('Enter your license key to unlock the plugin beyond the 30-day trial.', 'product-shortfacts-for-woocommerce'),
                'id' => 'psf_license_key',
            ],
            'psf_text_size' => [
                'name' => \__('Text Size', 'product-shortfacts-for-woocommerce'),
                'type' => 'select',
                'options' => [
                    'small' => \__('Small (12px)', 'product-shortfacts-for-woocommerce'),
                    'medium' => \__('Medium (14px)', 'product-shortfacts-for-woocommerce'),
                    'large' => \__('Large (16px)', 'product-shortfacts-for-woocommerce'),
                ],
                'default' => 'medium',
                'id' => 'psf_text_size',
            ],
            'psf_bullet_style' => [
                'name' => \__('List Style', 'product-shortfacts-for-woocommerce'),
                'type' => 'select',
                'options' => [
                    'bullet' => \__('Bulleted list', 'product-shortfacts-for-woocommerce'),
                    'check' => \__('Checkmarks list', 'product-shortfacts-for-woocommerce'),
                    'number' => \__('Numbered list', 'product-shortfacts-for-woocommerce'),
                ],
                'default' => 'bullet',
                'id' => 'psf_bullet_style',
            ],
            'psf_shortfacts_limit' => [
                'name' => \__('Shortfacts Limit', 'product-shortfacts-for-woocommerce'),
                'type' => 'number',
                'id' => 'psf_shortfacts_limit',
                'default' => 10,
                'desc' => \__('Maximum number of shortfacts per product (up to 10).', 'product-shortfacts-for-woocommerce'),
                'custom_attributes' => [ 'min' => 1, 'max' => 10 ],
            ],
            'psf_heading_text' => [
                'name' => \__('Heading Text', 'product-shortfacts-for-woocommerce'),
                'type' => 'text',
                'id' => 'psf_heading_text',
                'default' => 'Key Features',
                'desc' => \__('Customize the heading displayed above the shortfacts list.', 'product-shortfacts-for-woocommerce'),
            ],
            'psf_show_heading' => [
                'name' => \__('Show Heading', 'product-shortfacts-for-woocommerce'),
                'type' => 'checkbox',
                'id' => 'psf_show_heading',
                'default' => 'yes',
                'desc' => \__('Display the heading above the shortfacts list.', 'product-shortfacts-for-woocommerce'),
            ],
        ];
    }
    public static function show_license_status_notice(): void
    {
        if (!\current_user_can('manage_options')) {
            return;
        }
        // Prefer query arg, fallback to transient persisted in handler
        $st = null;
        $st_raw = \filter_input(INPUT_GET, 'psf_license_status');
        if (\is_string($st_raw)) {
            $st = \sanitize_text_field(\wp_unslash($st_raw));
        } else {
            $st = (string) \get_transient('psf_license_last_notice');
            if (!empty($st)) {
                \delete_transient('psf_license_last_notice');
            }
        }
        if (empty($st)) {
            return;
        }
        if ($st === 'valid') {
        echo '<div class="notice notice-success"><p>' . \esc_html__('License validated successfully.', 'product-shortfacts-for-woocommerce') . '</p></div>';
        } elseif ($st === 'invalid') {
        echo '<div class="notice notice-error"><p>' . \esc_html__('Invalid license key. Please check and try again.', 'product-shortfacts-for-woocommerce') . '</p></div>';
        } else {
        echo '<div class="notice notice-warning"><p>' . \esc_html__('License check skipped (network issue). Will retry later.', 'product-shortfacts-for-woocommerce') . '</p></div>';
        }
    }
}


