<?php
namespace wefathemes\externalmarketplaceslinksforwoocommerce;

if (!defined('ABSPATH')) {
    exit;
}

class Helpers
{
    public const MARKETPLACES = [
        'amazon', 'ebay', 'walmart', 'aliexpress', 'etsy', 'wayfair', 'rakuten', 'shopee', 'wildberries', 'ozon',
        'flipkart', 'jd', 'mercadolibre', 'coupang', 'trendyol', 'noon', 'allegro', 'cdiscount', 'lazada',
        'tokopedia', 'zalando', 'otto', 'target'
    ];

    public static function sanitize_url(string $url)
    {
        $url = trim($url);
        if ($url === '') {
            return '';
        }
        $sanitized = \esc_url_raw($url, ['http', 'https']);
        return $sanitized ?: '';
    }

    public static function validate_url(string $url): bool
    {
        if ($url === '') {
            return true; // empty allowed; handled as not set
        }
        return (bool) filter_var($url, FILTER_VALIDATE_URL) && preg_match('/^https?:\/\//i', $url);
    }

    public static function esc_attr($value)
    {
        return \esc_attr($value);
    }

    public static function get_marketplace_icon_path(string $marketplace): string
    {
        $filename = strtolower($marketplace) . '.svg';
        // Fallback to plugin assets only (WordPress plugin context)
        $asset_path = EXTEMALI_PLUGIN_DIR . 'assets/img/marketplaces/' . $filename;
        if (file_exists($asset_path)) {
            return $asset_path;
        }
        $default = EXTEMALI_PLUGIN_DIR . 'assets/img/marketplaces/default.svg';
        return $default;
    }

    public static function get_marketplace_icon_url(string $marketplace): ?string
    {
        $marketplace_lower = strtolower($marketplace);
        // Built-in mapping to reliable external logos (ported from CS-Cart module)
        $logos = [
            // Major global marketplaces - Clearbit CDN
            'amazon' => 'https://logo.clearbit.com/amazon.com',
            'ebay' => 'https://logo.clearbit.com/ebay.com',
            'walmart' => 'https://logo.clearbit.com/walmart.com',
            'aliexpress' => 'https://logo.clearbit.com/aliexpress.com',
            'etsy' => 'https://logo.clearbit.com/etsy.com',
            'rakuten' => 'https://logo.clearbit.com/rakuten.com',
            'mercadolibre' => 'https://logo.clearbit.com/mercadolibre.com',

            // Regional marketplaces - Clearbit CDN
            'wayfair' => 'https://logo.clearbit.com/wayfair.com',
            'shopee' => 'https://logo.clearbit.com/shopee.com',
            'flipkart' => 'https://logo.clearbit.com/flipkart.com',
            'jd' => 'https://logo.clearbit.com/jd.com',
            'coupang' => 'https://logo.clearbit.com/coupang.com',
            'zalando' => 'https://logo.clearbit.com/zalando.com',
            'lazada' => 'https://logo.clearbit.com/lazada.com',
            'tokopedia' => 'https://logo.clearbit.com/tokopedia.com',
            'allegro' => 'https://logo.clearbit.com/allegro.pl',
            'cdiscount' => 'https://logo.clearbit.com/cdiscount.com',

            // Custom inline SVGs for marketplaces without reliable external sources
            'wildberries' => 'data:image/svg+xml;base64,' . base64_encode('
                <svg width="40" height="40" xmlns="http://www.w3.org/2000/svg">
                    <rect width="40" height="40" fill="#cb11ab" rx="4"/>
                    <text x="20" y="26" text-anchor="middle" font-family="Arial, sans-serif" font-size="14" font-weight="bold" fill="white">WB</text>
                </svg>
            '),
            'ozon' => 'data:image/svg+xml;base64,' . base64_encode('
                <svg width="40" height="40" xmlns="http://www.w3.org/2000/svg">
                    <rect width="40" height="40" fill="#005bff" rx="4"/>
                    <text x="20" y="26" text-anchor="middle" font-family="Arial, sans-serif" font-size="12" font-weight="bold" fill="white">OZON</text>
                </svg>
            '),
            'trendyol' => 'data:image/svg+xml;base64,' . base64_encode('
                <svg width="40" height="40" xmlns="http://www.w3.org/2000/svg">
                    <rect width="40" height="40" fill="#f27a1a" rx="4"/>
                    <text x="20" y="26" text-anchor="middle" font-family="Arial, sans-serif" font-size="14" font-weight="bold" fill="white">TY</text>
                </svg>
            '),
            'noon' => 'data:image/svg+xml;base64,' . base64_encode('
                <svg width="40" height="40" xmlns="http://www.w3.org/2000/svg">
                    <rect width="40" height="40" fill="#feee00" rx="4"/>
                    <text x="20" y="26" text-anchor="middle" font-family="Arial, sans-serif" font-size="12" font-weight="bold" fill="#333">NOON</text>
                </svg>
            '),
            'otto' => 'data:image/svg+xml;base64,' . base64_encode('
                <svg width="40" height="40" xmlns="http://www.w3.org/2000/svg">
                    <rect width="40" height="40" fill="#d50000" rx="4"/>
                    <text x="20" y="26" text-anchor="middle" font-family="Arial, sans-serif" font-size="12" font-weight="bold" fill="white">OTTO</text>
                </svg>
            '),
            'target' => 'data:image/svg+xml;base64,' . base64_encode('
                <svg xmlns="http://www.w3.org/2000/svg" width="40" height="40" viewBox="0 0 40 40">
                    <circle cx="20" cy="20" r="18" fill="#CC0000"/>
                    <circle cx="20" cy="20" r="11" fill="#FFFFFF"/>
                    <circle cx="20" cy="20" r="6" fill="#CC0000"/>
                </svg>
            '),
        ];

        // If an external base URL is set, build the icon URL from it
        $base = trim(\get_option('extemali_external_icons_base_url', ''));
        if ($base !== '') {
            $base = \esc_url_raw($base, ['http', 'https']);
            if ($base !== '') {
                $base = rtrim($base, '/\\');
                $filename_svg = strtolower($marketplace) . '.svg';
                return $base . '/' . $filename_svg;
            }
        }
        // Built-in mapping fallback
        if (isset($logos[$marketplace_lower])) {
            return $logos[$marketplace_lower];
        }
        // Fallback to plugin assets only (WordPress plugin context)
        $assets_path = EXTEMALI_PLUGIN_DIR . 'assets/img/marketplaces/' . strtolower($marketplace) . '.svg';
        if (file_exists($assets_path)) {
        return EXTEMALI_PLUGIN_URL . 'assets/img/marketplaces/' . strtolower($marketplace) . '.svg';
        }
        // Default placeholder
        $default = EXTEMALI_PLUGIN_DIR . 'assets/img/marketplaces/default.svg';
        if (file_exists($default)) {
        return EXTEMALI_PLUGIN_URL . 'assets/img/marketplaces/default.svg';
        }
        return null;
    }
}