<?php
namespace wefathemes\externalmarketplaceslinksforwoocommerce;

if (!defined('ABSPATH')) {
    exit;
}

class Admin
{
    public static function init(): void
    {
        \add_action('add_meta_boxes', [self::class, 'add_product_metabox']);
        \add_action('save_post_product', [self::class, 'save_product_links'], 10, 2);
        \add_action('admin_enqueue_scripts', [self::class, 'enqueue_admin_assets']);

        // License prompt and handler
        \add_action('admin_notices', [self::class, 'render_license_prompt']);
        // Show post-submit license status notice regardless of license state
        \add_action('admin_notices', [self::class, 'show_license_status_notice']);
        \add_action('admin_post_extemali_save_license', [self::class, 'handle_license_submit']);

        // Modal overlay printed in admin footer when unlicensed
        \add_action('admin_footer', [self::class, 'render_license_modal']);

        \add_filter('woocommerce_settings_tabs_array', [self::class, 'register_settings_tab'], 50);
        \add_action('woocommerce_settings_tabs_extemali_settings', [self::class, 'render_settings']);
        \add_action('woocommerce_update_options_extemali_settings', [self::class, 'save_settings']);
    }

    public static function render_license_prompt(): void
    {
        // Show prompt to admins until licensed
        if (\function_exists('extemali_is_licensed') && \extemali_is_licensed()) {
            return;
        }
        if (!\current_user_can('manage_options')) {
            return;
        }
        $license_val = (string) \get_option('extemali_license_key', '');
        $action_url = \admin_url('admin-post.php');
        echo '<div class="notice notice-error"><form method="post" action="' . \esc_url($action_url) . '">';
        echo '<input type="hidden" name="action" value="extemali_save_license" />';
        \wp_nonce_field('extemali_save_license', 'extemali_save_license_nonce');
        echo '<p><strong>' . \esc_html__('External Marketplaces Links requires a license key', 'external-marketplaces-links-for-woocommerce') . '</strong></p>';
        echo '<p><label for="extemali_license_key_inline">' . \esc_html__('License Key', 'external-marketplaces-links-for-woocommerce') . '</label> ';
        echo '<input id="extemali_license_key_inline" type="text" name="extemali_license_key" class="regular-text" value="' . \esc_attr($license_val) . '" /> ';
        echo '<button type="submit" class="button button-primary">' . \esc_html__('Save and Validate', 'external-marketplaces-links-for-woocommerce') . '</button></p>';
        echo '<p>' . \esc_html__('You can buy a license at wefa.digital. This prompt will continue to appear until a valid license is saved.', 'external-marketplaces-links-for-woocommerce') . '</p>';
        echo '</form></div>';
    }

    public static function handle_license_submit(): void
    {
        if (!\current_user_can('manage_options')) {
            \wp_die('Unauthorized');
        }
        $nonce_val = \filter_input(INPUT_POST, 'extemali_save_license_nonce');
        $nonce_val = \is_string($nonce_val) ? \sanitize_text_field(\wp_unslash($nonce_val)) : '';
        if (empty($nonce_val) || !\wp_verify_nonce($nonce_val, 'extemali_save_license')) {
            \wp_die('Invalid nonce');
        }
        $key_raw = \filter_input(INPUT_POST, 'extemali_license_key');
        $key = \sanitize_text_field(\is_string($key_raw) ? \wp_unslash($key_raw) : '');
        $key = strtoupper(trim($key));
        \update_option('extemali_license_key', $key);
        $result = \extemali_check_license_valid_now();
        $status = ($result === true) ? 'valid' : (($result === false) ? 'invalid' : 'skipped');
        if ($status === 'valid') {
            \update_option('extemali_license_valid', 'yes');
        } elseif ($status === 'invalid') {
            \update_option('extemali_license_valid', 'no');
        }
        // Persist status temporarily to ensure a notice even if query args are stripped
        \set_transient('extemali_license_last_notice', $status, 300);
        $redirect = \wp_get_referer();
        if (!$redirect) {
            $redirect = \admin_url();
        }
        // Add a nonce to validate the origin of the status query param on redirect
        $status_nonce = \wp_create_nonce('extemali_license_status');
        // Surface a clear admin notice on the next page
        \wp_safe_redirect(\add_query_arg(['extemali_license_status' => $status, 'extemali_status_nonce' => $status_nonce], $redirect));
        exit;
    }

    public static function enqueue_admin_assets(): void
    {
        \wp_enqueue_style('extemali-admin', EXTEMALI_PLUGIN_URL . 'assets/admin.css', [], EXTEMALI_PLUGIN_VERSION);
        \wp_enqueue_script('extemali-admin', EXTEMALI_PLUGIN_URL . 'assets/admin.js', ['jquery'], EXTEMALI_PLUGIN_VERSION, true);
        $licensed = \function_exists('extemali_is_licensed') ? \extemali_is_licensed() : false;
        $reg = (int) \get_option('extemali_registration_date');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $expired = ($days_used >= 30);
        // Read status only if a valid nonce is present and user has capability
        $status = '';
        $status_raw = \filter_input(INPUT_GET, 'extemali_license_status');
        $nonce_raw = \filter_input(INPUT_GET, 'extemali_status_nonce');
        $nonce_val = \is_string($nonce_raw) ? \sanitize_text_field(\wp_unslash($nonce_raw)) : '';
        if (\current_user_can('manage_options') && \is_string($status_raw) && !empty($nonce_val) && \wp_verify_nonce($nonce_val, 'extemali_license_status')) {
            $status = \sanitize_text_field(\wp_unslash($status_raw));
        }
        \wp_localize_script('extemali-admin', 'EXTEMALIAdmin', [
            'licensed' => $licensed,
            'expired' => $expired,
            'status' => $status,
        ]);
    }

    public static function add_product_metabox(): void
    {
        // Only show metabox if licensed
        if (\function_exists('extemali_is_licensed') && !\extemali_is_licensed()) {
            return;
        }
        \add_meta_box(
            'extemali_marketplace_links',
            \__('External Marketplace Links', 'external-marketplaces-links-for-woocommerce'),
            [self::class, 'render_product_metabox'],
            'product',
            'normal',
            'default'
        );
    }

    public static function render_product_metabox($post): void
    {
        \wp_nonce_field('extemali_save_links', 'extemali_nonce');
        $links = \get_post_meta($post->ID, '_extemali_marketplace_links', true);
        if (!is_array($links) || empty($links)) {
            $links = \get_post_meta($post->ID, '_eml_marketplace_links', true);
        }
        if (!is_array($links)) {
            $links = [];
        }
        $display_type = \get_option('extemali_display_type', 'text');
        echo '<div class="eml-links-container">';
        echo '<p class="description">' . \esc_html__('Add links to your product listings on external marketplaces.', 'external-marketplaces-links-for-woocommerce') . '</p>';
        echo '<div class="eml-grid">';
        foreach (Helpers::MARKETPLACES as $marketplace) {
            $value = isset($links[$marketplace]) ? \esc_url($links[$marketplace]) : '';
            echo '<div class="eml-item">';
            echo '<label for="extemali_' . \esc_attr($marketplace) . '"><strong>' . \esc_html(ucfirst($marketplace)) . '</strong></label>';
            echo '<input type="url" class="regular-text" id="extemali_' . \esc_attr($marketplace) . '" name="extemali_links[' . \esc_attr($marketplace) . ']" value="' . \esc_attr($value) . '" placeholder="https://example.com/product" />';
            echo '</div>';
        }
        echo '</div>';
        echo '</div>';
    }

    public static function render_license_modal(): void
    {
        if (\function_exists('extemali_is_licensed') && \extemali_is_licensed()) {
            return;
        }
        if (!\current_user_can('manage_options')) {
            return;
        }
        $action_url = \admin_url('admin-post.php');
        $license_val = (string) \get_option('extemali_license_key', '');
        echo '<div id="eml-modal-overlay" class="eml-modal-overlay" aria-hidden="true">';
        echo '<div class="eml-modal" role="dialog" aria-labelledby="eml-modal-title" aria-modal="true">';
        echo '<div class="eml-modal-header">';
        // Add product heading above the license entry title
        echo '<div class="eml-title-group">';
        echo '<h3 class="eml-modal-subtitle">' . \esc_html__('External Marketplaces Links Pro', 'external-marketplaces-links-for-woocommerce') . '</h3>';
        echo '<h2 id="eml-modal-title">' . \esc_html__('Enter License Key', 'external-marketplaces-links-for-woocommerce') . '</h2>';
        echo '</div>';
        echo '<button type="button" id="eml-modal-close" class="button">' . \esc_html__('Close', 'external-marketplaces-links-for-woocommerce') . '</button>';
        echo '</div>';
        echo '<div class="eml-modal-body">';
        // Inline feedback inside modal based on last license attempt
        $st_raw = \filter_input(INPUT_GET, 'extemali_license_status');
        $nonce_raw = \filter_input(INPUT_GET, 'extemali_status_nonce');
        $nonce_val = \is_string($nonce_raw) ? \sanitize_text_field(\wp_unslash($nonce_raw)) : '';
        if (\is_string($st_raw) && !empty($nonce_val) && \wp_verify_nonce($nonce_val, 'extemali_license_status')) {
            $st = \sanitize_text_field(\wp_unslash($st_raw));
            if ($st === 'valid') {
                echo '<div class="notice notice-success"><p>' . \esc_html__('License validated successfully.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            } elseif ($st === 'invalid') {
                echo '<div class="notice notice-error"><p>' . \esc_html__('Invalid license key. Please check and try again.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            } elseif ($st === 'skipped') {
                echo '<div class="notice notice-warning"><p>' . \esc_html__('License check skipped (network issue). Will retry later.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            }
        }
        echo '<p>' . \esc_html__('Your trial has expired. Enter your license key to continue using External Marketplaces Links.', 'external-marketplaces-links-for-woocommerce') . '</p>';
        echo '<form method="post" action="' . \esc_url($action_url) . '" class="eml-license-form">';
        echo '<input type="hidden" name="action" value="extemali_save_license" />';
        \wp_nonce_field('extemali_save_license', 'extemali_save_license_nonce');
        echo '<label for="extemali_license_key_modal"><strong>' . \esc_html__('License Key', 'external-marketplaces-links-for-woocommerce') . '</strong></label>';
        echo '<input id="extemali_license_key_modal" type="text" name="extemali_license_key" class="regular-text" value="' . \esc_attr($license_val) . '" placeholder="WEFAEML-XXXX-XXXX-XXXX-XXXX" />';
        echo '<div class="eml-modal-actions">';
        echo '<button type="submit" class="button button-primary">' . \esc_html__('Save and Validate', 'external-marketplaces-links-for-woocommerce') . '</button>';
        echo '<a href="https://wefa.digital/wp-ext-marketplaces/" target="_blank" rel="noopener" class="button btn-danger">' . \esc_html__('Buy License', 'external-marketplaces-links-for-woocommerce') . '</a>';
        echo '</div>';
        echo '</form>';
        echo '</div>';
        echo '</div>';
        echo '</div>';
    }

    public static function save_product_links($post_id, $post): void
    {
        if (\function_exists('extemali_is_licensed') && !\extemali_is_licensed()) {
            return;
        }
        $nonce_links = \filter_input(INPUT_POST, 'extemali_nonce');
        $nonce_links = \is_string($nonce_links) ? \sanitize_text_field(\wp_unslash($nonce_links)) : '';
        if (empty($nonce_links) || !\wp_verify_nonce($nonce_links, 'extemali_save_links')) {
            return;
        }
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if ($post->post_type !== 'product') {
            return;
        }
        if (!\current_user_can('edit_post', $post_id)) {
            return;
        }
        $raw = \filter_input(INPUT_POST, 'extemali_links', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY);
        $raw = \is_array($raw) ? \wp_unslash($raw) : [];
        $sanitized = [];
        foreach (Helpers::MARKETPLACES as $marketplace) {
            $value = isset($raw[$marketplace]) ? \sanitize_text_field($raw[$marketplace]) : '';
            $value = Helpers::sanitize_url($value);
            if ($value !== '' && Helpers::validate_url($value)) {
                $sanitized[$marketplace] = $value;
            }
        }
        \update_post_meta($post_id, '_extemali_marketplace_links', $sanitized);
    }

    public static function register_settings_tab($tabs)
    {
        $tabs['extemali_settings'] = \__('Marketplace Links', 'external-marketplaces-links-for-woocommerce');
        return $tabs;
    }

    public static function render_settings(): void
    {
        if (!\current_user_can('manage_options')) {
            return;
        }
        echo '<div class="eml-settings eml-card">';
        echo '<h2>' . \esc_html__('External Marketplace Links Settings', 'external-marketplaces-links-for-woocommerce') . '</h2>';

        // Status after inline save
        $st_raw_settings = \filter_input(INPUT_GET, 'extemali_license_status');
        $nonce_raw = \filter_input(INPUT_GET, 'extemali_status_nonce');
        $nonce_val = \is_string($nonce_raw) ? \sanitize_text_field(\wp_unslash($nonce_raw)) : '';
        if (\is_string($st_raw_settings) && !empty($nonce_val) && \wp_verify_nonce($nonce_val, 'extemali_license_status')) {
            $st = \sanitize_text_field(\wp_unslash($st_raw_settings));
            if ($st === 'valid') {
                echo '<div class="notice notice-success"><p>' . \esc_html__('License validated successfully.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            } elseif ($st === 'invalid') {
                echo '<div class="notice notice-error"><p>' . \esc_html__('Invalid license key. Please check and try again.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            } else {
                echo '<div class="notice notice-warning"><p>' . \esc_html__('License check skipped (network issue). Will retry later.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            }
        }

        // Trial info and license
        $reg = (int) \get_option('extemali_registration_date');
        $license = (string) \get_option('extemali_license_key');
        $now = \current_time('timestamp');
        $days_used = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 0;
        $days_left = max(0, 30 - $days_used);
        $license_status = \get_option('extemali_license_last_status');
        $license_valid = \get_option('extemali_license_valid') === 'yes';
        $expired = ($days_used >= 30);
        $is_licensed = (\function_exists('extemali_is_licensed')) ? \extemali_is_licensed() : $license_valid;

        if (! $is_licensed) {
            if ($expired) {
                echo '<div class="notice notice-error"><p>' . \esc_html__('Your trial has expired. Enter your license key to unlock External Marketplaces Links.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            } else {
                echo '<div class="notice notice-warning"><p>' . \esc_html__('Please enter a valid license key to unlock External Marketplaces Links.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
            }
        } else {
            echo '<div class="eml-settings-row">';
            echo '<div class="eml-col">';
            echo '<p><strong>' . \esc_html__('Registration Date:', 'external-marketplaces-links-for-woocommerce') . '</strong> ' . ($reg ? \esc_html(\date_i18n(\get_option('date_format'), $reg)) : \esc_html__('Not set', 'external-marketplaces-links-for-woocommerce')) . '</p>';
            // Hide Trial Days Left once licensed
            if (!empty($license)) {
                echo '<p><strong>' . \esc_html__('License Status:', 'external-marketplaces-links-for-woocommerce') . '</strong> ' . \esc_html($license_valid ? \__('Valid', 'external-marketplaces-links-for-woocommerce') : ($license_status === 'skipped' ? \__('Skipped today', 'external-marketplaces-links-for-woocommerce') : \__('Unknown', 'external-marketplaces-links-for-woocommerce'))) . '</p>';
            }
            echo '<p><strong>' . \esc_html__('Support:', 'external-marketplaces-links-for-woocommerce') . '</strong> <a href="mailto:shop@wefa.digital">shop@wefa.digital</a></p>';
            echo '</div>';
            echo '<div class="eml-col">';
        echo '<a class="button button-primary" target="_blank" rel="noopener" href="https://wefa.digital/wp-ext-marketplaces/">' . \esc_html__('Buy Pro Version on wefa.digital', 'external-marketplaces-links-for-woocommerce') . '</a>';
            echo '</div>';
            echo '</div>';
        }

        // Keep display fields synced
        $formatted_date = $reg ? \date_i18n(\get_option('date_format'), $reg) : \__('Not set', 'external-marketplaces-links-for-woocommerce');
        \update_option('extemali_registration_date_display', $formatted_date);
        \update_option('extemali_trial_days_left', $license_valid ? '' : (string) $days_left);
        \update_option('extemali_support_email_display', 'shop@wefa.digital');

        // Render settings fields
        $fields = self::get_settings_fields();
        if (! $is_licensed) {
            // Hide all configuration; show only license field when unlicensed
            $fields = [
                'section_title' => [
                    'name' => \__('External Marketplace Links', 'external-marketplaces-links-for-woocommerce'),
                    'type' => 'title',
                    'desc' => \__('Enter your license key to unlock the plugin.', 'external-marketplaces-links-for-woocommerce'),
                    'id' => 'extemali_section_title',
                ],
                'extemali_license_key' => $fields['extemali_license_key'],
            ];
        } else {
            // When licensed, hide Trial Days Left field from settings
            if (isset($fields['extemali_trial_days_left'])) {
                unset($fields['extemali_trial_days_left']);
            }
        }
        \woocommerce_admin_fields($fields);
        echo '</div>';
    }

    public static function save_settings(): void
    {
        \woocommerce_update_options(self::get_settings_fields());
    }

    private static function get_settings_fields(): array
    {
        return [
            'section_title' => [
                'name' => \__('External Marketplace Links', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'title',
                'desc' => \__('Configure how marketplace links are displayed.', 'external-marketplaces-links-for-woocommerce'),
                'id' => 'extemali_section_title',
            ],
            'extemali_registration_date_display' => [
                'name' => \__('Registration Date', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'text',
                'id' => 'extemali_registration_date_display',
                'desc_tip' => true,
                'desc' => \__('Automatically set on activation.', 'external-marketplaces-links-for-woocommerce'),
                'custom_attributes' => [ 'disabled' => 'disabled' ],
            ],
            'extemali_trial_days_left' => [
                'name' => \__('Trial Days Left', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'text',
                'id' => 'extemali_trial_days_left',
                'custom_attributes' => [ 'disabled' => 'disabled' ],
            ],
            'extemali_support_email_display' => [
                'name' => \__('Support Email', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'text',
                'id' => 'extemali_support_email_display',
                'custom_attributes' => [ 'disabled' => 'disabled' ],
            ],
            'extemali_license_key' => [
                'name' => \__('License Key', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'text',
                'desc' => \__('Enter your license key to unlock the plugin beyond the 30-day trial.', 'external-marketplaces-links-for-woocommerce'),
                'id' => 'extemali_license_key',
            ],
            'extemali_display_type' => [
                'name' => \__('Display Type', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'select',
                'options' => [
                    'text' => \__('Text Buttons', 'external-marketplaces-links-for-woocommerce'),
                    'image' => \__('Image Logos', 'external-marketplaces-links-for-woocommerce'),
                ],
                'default' => 'text',
                'id' => 'extemali_display_type',
            ],
            'extemali_image_size' => [
                'name' => \__('Image Size', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'select',
                'options' => [
                    'small' => \__('Small (24px)', 'external-marketplaces-links-for-woocommerce'),
                    'medium' => \__('Medium (32px)', 'external-marketplaces-links-for-woocommerce'),
                    'large' => \__('Large (48px)', 'external-marketplaces-links-for-woocommerce'),
                ],
                'default' => 'medium',
                'id' => 'extemali_image_size',
            ],
            'extemali_text_size' => [
                'name' => \__('Text Size', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'select',
                'options' => [
                    'small' => \__('Small (10px)', 'external-marketplaces-links-for-woocommerce'),
                    'medium' => \__('Medium (14px)', 'external-marketplaces-links-for-woocommerce'),
                    'large' => \__('Large (16px)', 'external-marketplaces-links-for-woocommerce'),
                ],
                'default' => 'medium',
                'id' => 'extemali_text_size',
            ],
            'extemali_show_text_with_images' => [
                'name' => \__('Show Text with Images', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'checkbox',
                'default' => 'yes',
                'id' => 'extemali_show_text_with_images',
            ],
            'extemali_external_icons_base_url' => [
                'name' => \__('External Icons Base URL', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'text',
                'id' => 'extemali_external_icons_base_url',
                'desc' => \__('If set, marketplace icons are loaded from this base URL using filenames like amazon.svg, ebay.svg. If not set, the plugin uses built-in mappings (e.g., Clearbit logos) or local assets as fallback. Example: https://cdn.example.com/marketplaces', 'external-marketplaces-links-for-woocommerce'),
            ],
            'extemali_heading_text' => [
                'name' => \__('Heading Text', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'text',
                'id' => 'extemali_heading_text',
                'default' => 'Buy on Marketplaces',
                'desc' => \__('Customize the heading displayed above the marketplace links.', 'external-marketplaces-links-for-woocommerce'),
            ],
            'extemali_show_heading' => [
                'name' => \__('Show Heading', 'external-marketplaces-links-for-woocommerce'),
                'type' => 'checkbox',
                'id' => 'extemali_show_heading',
                'default' => 'yes',
                'desc' => \__('Display the heading above the marketplace links.', 'external-marketplaces-links-for-woocommerce'),
            ],
        ];
    }
    public static function show_license_status_notice(): void
    {
        if (!\current_user_can('manage_options')) {
            return;
        }
        // Prefer query arg, fallback to transient persisted in handler
        $st = null;
        $st_raw = \filter_input(INPUT_GET, 'extemali_license_status');
        $nonce_raw = \filter_input(INPUT_GET, 'extemali_status_nonce');
        $nonce_val = \is_string($nonce_raw) ? \sanitize_text_field(\wp_unslash($nonce_raw)) : '';
        if (\is_string($st_raw) && !empty($nonce_val) && \wp_verify_nonce($nonce_val, 'extemali_license_status')) {
            $st = \sanitize_text_field(\wp_unslash($st_raw));
        } else {
            $st = (string) \get_transient('extemali_license_last_notice');
            if (!empty($st)) {
                \delete_transient('extemali_license_last_notice');
            }
        }
        if (empty($st)) {
            return;
        }
        if ($st === 'valid') {
        echo '<div class="notice notice-success"><p>' . \esc_html__('License validated successfully.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
        } elseif ($st === 'invalid') {
        echo '<div class="notice notice-error"><p>' . \esc_html__('Invalid license key. Please check and try again.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
        } else {
        echo '<div class="notice notice-warning"><p>' . \esc_html__('License check skipped (network issue). Will retry later.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
        }
    }
}