<?php
/**
 * Plugin Name: External Marketplaces Links for WooCommerce
 * Description: Add links to product listings on external marketplaces and display them on product pages with configurable styles.
 * Version: 1.0.24
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * Author: wefathemes
 * Author URI: https://wefa.digital/wp-ext-marketplaces/
 * Text Domain: external-marketplaces-links-for-woocommerce
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define constants (unique, vendor-prefixed)
define('EXTEMALI_PLUGIN_VERSION', '1.0.24');
define('EXTEMALI_PLUGIN_FILE', __FILE__);
define('EXTEMALI_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('EXTEMALI_PLUGIN_URL', plugin_dir_url(__FILE__));

// Ensure WooCommerce is active
function extemali_is_woocommerce_active(): bool {
    return in_array('woocommerce/woocommerce.php', (array) get_option('active_plugins', []), true)
        || (function_exists('is_plugin_active') && is_plugin_active('woocommerce/woocommerce.php'));
}

// Activation: basic checks
function extemali_activate() {
    if (!extemali_is_woocommerce_active()) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(esc_html__('WooCommerce must be active to use External Marketplaces Links.', 'external-marketplaces-links-for-woocommerce'));
    }
    // Initialize registration date if not set
    if (!get_option('extemali_registration_date')) {
        update_option('extemali_registration_date', current_time('timestamp'));
    }
}
register_activation_hook(__FILE__, 'extemali_activate');

// Deactivate after 30-day trial ends
function extemali_maybe_deactivate_if_expired() {
    $reg = (int) get_option('extemali_registration_date');
    $now = current_time('timestamp');
    $days = $reg ? floor(($now - $reg) / DAY_IN_SECONDS) : 9999; // Treat missing date as expired
    if ($days >= 30) {
        // If license already valid, nothing to do
        $licensed = get_option('extemali_license_valid') === 'yes';
        if ($licensed) {
            return;
        }
        // Show a non-blocking admin notice encouraging license entry (modal will auto-open)
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>' . esc_html__('External Marketplaces Links trial has expired. Please enter your license key to continue using the plugin.', 'external-marketplaces-links-for-woocommerce') . '</p></div>';
        });
    }
}
add_action('admin_init', 'extemali_maybe_deactivate_if_expired');

// Autoload simple includes
require_once EXTEMALI_PLUGIN_DIR . 'includes/helpers.php';
require_once EXTEMALI_PLUGIN_DIR . 'includes/admin.php';
require_once EXTEMALI_PLUGIN_DIR . 'includes/frontend.php';

// i18n: WordPress loads translations automatically from language packs.
// No manual textdomain loading is needed or recommended.

// Admin init
add_action('plugins_loaded', function () {
    if (!extemali_is_woocommerce_active()) {
        return;
    }
    \wefathemes\externalmarketplaceslinksforwoocommerce\Admin::init();
    // Initialize frontend only if licensed
    $licensed = get_option('extemali_license_valid') === 'yes';
    if ($licensed) {
        \wefathemes\externalmarketplaceslinksforwoocommerce\Frontend::init();
    }
});

// Ensure defaults if missing (handles cases where activation hook did not set options)
function extemali_ensure_defaults() {
    // Create license key option if missing
    $license_opt = get_option('extemali_license_key', null);
    if ($license_opt === null) {
        add_option('extemali_license_key', '');
    }
}
add_action('admin_init', 'extemali_ensure_defaults');

// Helper: is plugin licensed
function extemali_is_licensed(): bool {
    $key = trim((string) get_option('extemali_license_key'));
    if ($key === '') {
        return false;
    }
    return get_option('extemali_license_valid') === 'yes';
}

// Sync: if license key becomes empty, ensure valid flag is reset
function extemali_sync_license_valid_flag() {
    $key = trim((string) get_option('extemali_license_key'));
    if ($key === '' && get_option('extemali_license_valid') === 'yes') {
        update_option('extemali_license_valid', 'no');
    }
}
add_action('init', 'extemali_sync_license_valid_flag');
add_action('admin_init', 'extemali_sync_license_valid_flag');

// License validation
function extemali_check_license_valid_now(): ?bool {
    $license_raw = trim((string) get_option('extemali_license_key'));
    // Normalize license to uppercase to match expected format on server
    $license = strtoupper($license_raw);
    // Basic format validation before remote call
    if ($license === '' || !preg_match('/^WEFAEML-[A-Z0-9]{8}-[A-Z0-9]{8}-[A-Z0-9]{8}-[A-Z0-9]{8}$/', $license)) {
        update_option('extemali_license_last_status', 'invalid');
        update_option('extemali_license_last_checked', current_time('timestamp'));
        update_option('extemali_license_valid', 'no');
        return false;
    }
    $url = add_query_arg(['license' => rawurlencode($license)], 'https://wefa.digital/wp_ext_mp_licence_check.php');
    $resp = wp_remote_get($url, [
        'timeout' => 5,
        'redirection' => 3,
        'headers' => [ 'Accept' => 'text/plain' ],
    ]);
    if (is_wp_error($resp)) {
        update_option('extemali_license_last_status', 'skipped');
        update_option('extemali_license_last_checked', current_time('timestamp'));
        return null;
    }
    $code = wp_remote_retrieve_response_code($resp);
    $body = trim((string) wp_remote_retrieve_body($resp));
    update_option('extemali_license_last_checked', current_time('timestamp'));
    if ((int) $code === 200 && $body === 'valid licence !') {
        update_option('extemali_license_last_status', 'valid');
        update_option('extemali_license_valid', 'yes');
        return true;
    } elseif ((int) $code === 403 || $body === 'invalid licence') {
        update_option('extemali_license_last_status', 'invalid');
        update_option('extemali_license_valid', 'no');
        return false;
    }
    update_option('extemali_license_last_status', 'skipped');
    return null;
}

function extemali_maybe_check_license_daily() {
    $license = trim((string) get_option('extemali_license_key'));
    if (empty($license)) {
        return;
    }
    $last = (int) get_option('extemali_license_last_checked');
    $now = current_time('timestamp');
    if (!$last || ($now - $last) >= DAY_IN_SECONDS) {
        extemali_check_license_valid_now();
    }
}
add_action('admin_init', 'extemali_maybe_check_license_daily');

// Migrate legacy option/transient keys to new prefix, once
function extemali_migrate_options_and_transients() {
    $map = [
        'eml_registration_date' => 'extemali_registration_date',
        'eml_license_key' => 'extemali_license_key',
        'eml_license_valid' => 'extemali_license_valid',
        'eml_license_last_status' => 'extemali_license_last_status',
        'eml_license_last_checked' => 'extemali_license_last_checked',
        'eml_registration_date_display' => 'extemali_registration_date_display',
        'eml_trial_days_left' => 'extemali_trial_days_left',
        'eml_support_email_display' => 'extemali_support_email_display',
        'eml_display_type' => 'extemali_display_type',
        'eml_image_size' => 'extemali_image_size',
        'eml_text_size' => 'extemali_text_size',
        'eml_show_text_with_images' => 'extemali_show_text_with_images',
        'eml_external_icons_base_url' => 'extemali_external_icons_base_url',
        'eml_heading_text' => 'extemali_heading_text',
        'eml_show_heading' => 'extemali_show_heading',
    ];
    foreach ($map as $old => $new) {
        $old_val = get_option($old, null);
        $new_val = get_option($new, null);
        if ($old_val !== null && $new_val === null) {
            update_option($new, $old_val);
            delete_option($old);
        }
    }
    $t_old = get_transient('eml_license_last_notice');
    if ($t_old !== false && get_transient('extemali_license_last_notice') === false) {
        set_transient('extemali_license_last_notice', $t_old, 300);
        delete_transient('eml_license_last_notice');
    }
}
add_action('plugins_loaded', 'extemali_migrate_options_and_transients', 1);